////////////////////////////////////////////////////////////////////////////////
//
// Copyright (c) since 1999. VASCO DATA SECURITY
// All rights reserved. http://www.vasco.com
//
////////////////////////////////////////////////////////////////////////////////

#import <Foundation/Foundation.h>
NS_ASSUME_NONNULL_BEGIN

@class CDDCParams;
@protocol OrchestrationProtocol;
@protocol OrchestrationErrorDelegate;
@protocol OrchestrationWarningDelegate;
@protocol OrchestrationAccessGroup;

/**
 * Simple builder for the {@link OrchestrationProtocol} object.
 */
NS_REFINED_FOR_SWIFT
@interface OrchestratorBuilder : NSObject

/**
 * Returns the current version of MSSOrchestration
 */
@property (class, nonatomic, readonly) const NSString *sdkVersion;

/**
 * build an {@link OrchestrationProtocol} object with the configured {@link OrchestratorBuilder} instance.
 * @param error a NSError raised if there is an error during the process.
 */
- (id<OrchestrationProtocol> _Nullable)buildWithError:(NSError **)error;

/**
 * Set the salt that is used to protect the secret of the DIGIPASS and return the modified {@link OrchestratorBuilder}.
 * @param digipassSalt digipass salt
 * @return modified {@link OrchestratorBuilder}
 */
- (OrchestratorBuilder *)setDigipassSalt:(NSString *)digipassSalt;

/**
 * Set the salt that is used to protect the storage and return the modified {@link OrchestratorBuilder}.
 * @param storageSalt storage salt
 * @return modified {@link OrchestratorBuilder}
 */
- (OrchestratorBuilder *)setStorageSalt:(NSString *)storageSalt;

/**
 * Set the access group that will be use to bind the device
 * and return the modified {@link OrchestratorBuilder}.
 * @param accessGroup {@link OrchestrationAccessGroup}
 * @return modified {@link OrchestratorBuilder}
 */
- (OrchestratorBuilder *)setAccessGroup:(id<OrchestrationAccessGroup>)accessGroup;

/**
 * Sets the domain corresponding to the current user's account.
 * @param defaultDomain the unique identifier of the domain related to the user's account (e.g. "VASCO")
 */
- (OrchestratorBuilder *)setDefaultDomain:(NSString *)defaultDomain;

/**
 * Set the CDDC parameters. These parameters are used during CDDDC process and return the modified {@link OrchestratorBuilder}.
 * @param cddcParams CDDC parameters used to configure CDDC process
 * @return modified {@link OrchestratorBuilder}
 */
- (OrchestratorBuilder *)setCDDCParams:(CDDCParams *)cddcParams;

/**
 * Set the delegate that will be used if an error occurs and return the modified {@link OrchestratorBuilder}.
 * @param orchestrationErrorDelegate error delegate
 * @return modified {@link OrchestratorBuilder}
 */
- (OrchestratorBuilder *)setErrorDelegate:(id<OrchestrationErrorDelegate>)orchestrationErrorDelegate;

/**
 * Set the delegate that will be used if a warning occurs. The method then returns the modified {@link OrchestratorBuilder}.
 * @param orchestrationErrorDelegate error delegate
 * @return modified {@link OrchestratorBuilder}
 */
- (OrchestratorBuilder *)setWarningDelegate:(id <OrchestrationWarningDelegate>)orchestrationWarningDelegate;

@end

NS_ASSUME_NONNULL_END
